/***********************************************************************************************************/
/*   This code (© 2015 Thomas Jefferson University, All Rights Reserved) was created by Venetia Pliatsika  */
/*   and Isidore Rigoutsos and is an implementation of the Off-Spotter algorithm that appears in           */
/*   Pliatsika, V, and Rigoutsos, I (2015) "Off-Spotter: very fast and exhaustive enumeration of genomic   */
/*   lookalikes for designing CRISPR/Cas guide RNAs" Biol. Direct 10(1):4.                                 */
/*                                                                                                         */
/* Use of these codes is bound by the following terms and conditions:                                      */
/*                                                                                                         */
/* Terms of Use: This code can be freely used for research, academic and other non-profit activities.      */
/* Only one instance of the code may be used at a time, and then for only one concurrent user. You may not */
/* use the code to conduct any type of application service, service bureau or time-sharing operation or to */
/* provide any remote processing, network processing, network telecommunications or similar services to    */
/* any person, entity or organization, whether on a fee basis or otherwise. The code can be copied and     */
/* compiled on any platform for the use authorized by these terms and conditions. All copies of the code   */
/* must be accompanied by this note. The code cannot be modified without the written permission of the     */
/* Computational Medicine Center of Thomas Jefferson University https://cm.jefferson.edu                   */
/*                                                                                                         */
/* Commercial use is strictly prohibited.  If you wish to use these codes commercially please contact the  */
/* Computational Medicine Center of Thomas Jefferson University: https://cm.jefferson.edu/contact-us/      */
/*                                                                                                         */
/*                                                                                                         */
/*     THE CODE IS PROVIDED “AS IS” WITH NO REPRESENTATIONS OR WARRANTIES OF ANY KIND, EITHER EXPRESSED    */
/*     OR IMPLIED. TO THE FULLEST EXTENT PERMISSIBLE PURSUANT TO APPLICABLE LAW. THOMAS JEFFERSON          */
/*     UNIVERSITY, AND ITS AFFILIATES, DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING, BUT NOT     */
/*     LIMITED TO, THE IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND  */
/*     NON-INFRINGEMENT.                                                                                   */
/*                                                                                                         */
/*     NEITHER THOMAS JEFFERSON UNIVERSITY NOR ITS AFFILIATES MAKE ANY REPRESENTATION AS TO THE RESULTS    */
/*     TO BE OBTAINED FROM USE OF THE CODE.                                                                */
/***********************************************************************************************************/

#include <iostream>
#include <fstream>
#include <cstring>
#include <cstdlib>
#include <ctime>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <cerrno>

#include "Off-Spotter_Shared_Memory_IDs.h"
#include "Off-Spotter_General.h"

using namespace std;

//Table A, index table
unsigned int *hashTable;
//Table B, result table
MYTYPE *tableValues;
//Size of table B (the size of A is fixed and in Off-Spotter_General.h)
unsigned int *sizes;


void readTables (char *FILEINDEX, char *FILEDATA){
    FILE *f;
    struct stat fileStat;
    
    //Allocate and attach shared memory for the size of table B
    int sizes_ID = shmget(SIZESKEY,sizeof(unsigned int),IPC_CREAT | 0644);
    sizes = (unsigned int *) shmat(sizes_ID,NULL,0);
    
    //open file with table A
    cout << "\nLoading Table A, the index table. Please wait." << endl;
    f = fopen ( FILEINDEX , "rb" );
    //allocate shared memory to contain the read data
    int hashID = shmget(HASHKEY,KEY_NUM*sizeof(unsigned int),IPC_CREAT | 0644);
    //Attach shared memory
    hashTable = (unsigned int *) shmat(hashID,NULL,0);
    //read index and table
    fread(hashTable,sizeof(unsigned int),KEY_NUM,f);
    fclose(f);
    
    //open file with table B
    cout << "\nLoading Table B, the data table. Please wait." << endl;
    f = fopen (FILEDATA , "rb" );
    //obtain file size and assign it to the size shared memory segment
    stat(FILEDATA,&fileStat);
    sizes[0] = fileStat.st_size/sizeof(MYTYPE);
    //allocate space in memory for table B
    int dataID = shmget(DATAKEY,sizes[0]*sizeof(MYTYPE),IPC_CREAT | 0644);
    //attach table B
    tableValues = (MYTYPE *) shmat(dataID,NULL,0);
    //read table B on memory and close file
    fread(tableValues,sizeof(MYTYPE),sizes[0],f);
    fclose(f);
}


int main (int argc, char *argv[]){
    //Input must include (only) the path to the directory where index.bin and data.bin are located
    //Both must be on the same directory
    if ( argc != 2 ) {
        fprintf(stderr, "\n\nUsage: %s <input_folder_full_path>\n\n", argv[0]);
        fflush(stderr);
        exit(1);
    }
    
    //Create the input filenames
    char *in_file_path = argv[1];
    char index_file[strlen(in_file_path) + strlen("/index.bin") + 1], data_file[strlen(in_file_path) + strlen("/data.bin") + 1];
    strcpy(index_file,in_file_path);
    strcpy(data_file,in_file_path);
    if (in_file_path[strlen(in_file_path) - 1] == '/'){
        strcat(index_file,"index.bin");
        strcat(data_file,"data.bin");
    }
    else{
        strcat(index_file,"/index.bin");
        strcat(data_file,"/data.bin");
    }

    //Load tha tables (files) in memory
    readTables(index_file,data_file);
    
    //detach
    shmdt(sizes);
    shmdt(hashTable);
    shmdt(tableValues);
    
    cout << "\nTables loaded. You can use \"Results\" now.\n" << endl;
    
    return 0;
}
